// js/fsk.js - player management and score tracking for Farkle Score Keeper

const STORAGE_PREFIX = 'fsk_game_';
const STORAGE_LIST_KEY = 'fsk_game_list';
let players = [];
let currentPlayerId = null;
let gameId = null;
let ignoreNextBlur = false;
let savedGames = [];
let deleteTargetGameId = null;

 // Final round state: when a player reaches >=10000, enter final round.
 // finalRoundStarterId is the player who triggered the final round.
 // finalRoundReached tracks players who have reached the threshold during the final round.
 let finalRound = false;
 let finalRoundStarterId = null;
 let finalRoundReached = new Set();
 // finalRoundFinished tracks players who have completed their final turn (including the starter).
 let finalRoundFinished = new Set();
 // gameEnded indicates a finished game; savedWinners stores a small snapshot persisted with the game.
 let gameEnded = false;
 let savedWinners = [];

function loadGameList() {
  try {
    const raw = localStorage.getItem(STORAGE_LIST_KEY);
    savedGames = raw ? JSON.parse(raw) : [];
  } catch (e) {
    savedGames = [];
  }
}

function saveGameList() {
  localStorage.setItem(STORAGE_LIST_KEY, JSON.stringify(savedGames));
}

function loadPlayers(gameKey) {
  try {
    const raw = localStorage.getItem(gameKey);
    const data = raw ? JSON.parse(raw) : { players: [], currentPlayerId: null, gameEnded: false, savedWinners: [] };
    players = data.players || [];
    currentPlayerId = data.currentPlayerId || (players.length ? players[0].id : null);
    gameId = gameKey;
    gameEnded = !!data.gameEnded;
    savedWinners = data.savedWinners || [];
  } catch (e) {
    players = [];
    currentPlayerId = null;
    gameId = null;
    gameEnded = false;
    savedWinners = [];
  }
}

function savePlayers() {
  if (!gameId) return;
  // Persist game state including whether the game has ended and a small winners snapshot
  localStorage.setItem(gameId, JSON.stringify({ players, currentPlayerId, gameEnded: !!gameEnded, savedWinners }));
}

function startNewGame(name) {
  // Reset end-game related flags in case a previous ended game was shown
  gameEnded = false;
  savedWinners = [];
  finalRound = false;
  finalRoundStarterId = null;
  finalRoundReached = new Set();
  finalRoundFinished = new Set();

  // Generate a new game id
  gameId = STORAGE_PREFIX + Date.now().toString(36) + Math.random().toString(36).slice(2,7);
  players = [];
  currentPlayerId = null;
  const gameName = name && name.trim() ? name.trim() : ('Game - ' + new Date().toLocaleString());
  savedGames.push({ id: gameId, name: gameName });
  saveGameList();
  savePlayers();
  showGameUI();
}

function showGameUI() {
  // hide landing controls and show game UI
  const startBtn = document.getElementById('start-new-game-btn');
  const loadBtn = document.getElementById('load-game-btn');
  if (startBtn) startBtn.style.display = 'none';
  if (loadBtn) loadBtn.style.display = 'none';
  const landing = document.getElementById('landing-screen');
  if (landing) {
    // fully hide landing screen from layout and accessibility tree
    landing.style.display = 'none';
    landing.hidden = true;
    landing.setAttribute('aria-hidden', 'true');
    // reduce any min-height so it doesn't reserve vertical space
    landing.style.minHeight = '0px';
  }
  document.getElementById('game-ui').style.display = '';
  // mark body as in-game to allow CSS adjustments
  document.body.classList.add('in-game');
  // hide the default initializing status text once the game UI is shown
  const statusEl = document.getElementById('status');
  if (statusEl) statusEl.style.display = 'none';

  // Re-enable interactive controls in case they were previously disabled
  // (e.g., when viewing an ended game). This ensures adding players and using
  // the score inputs works after starting a new game or loading a live game.
  try {
    const controls = document.querySelectorAll('#players-tbody input, #players-tbody button, #add-player-form input, #add-player-form button');
    controls.forEach(el => el.disabled = false);
  } catch (e) {
    // ignore
  }

  renderPlayers();
}

function showLandingScreen() {
  // restore landing controls and hide game UI
  const startBtn = document.getElementById('start-new-game-btn');
  const loadBtn = document.getElementById('load-game-btn');
  if (startBtn) startBtn.style.display = '';
  if (loadBtn) loadBtn.style.display = '';
  const landing = document.getElementById('landing-screen');
  if (landing) {
    landing.style.display = '';
    landing.hidden = false;
    landing.setAttribute('aria-hidden', 'false');
    // restore default min-height
    landing.style.minHeight = '300px';
  }
  document.getElementById('game-ui').style.display = 'none';
  // remove in-game marker so spacing returns to normal
  document.body.classList.remove('in-game');
}

function addPlayer(name) {
  if (!name) return;
  const id = Date.now().toString(36) + Math.random().toString(36).slice(2,7);
  players.push({ id, name: name.trim(), score: 0, farkles: 0 });
  // if this is the first player, set as current
  if (!currentPlayerId) currentPlayerId = id;
  savePlayers();
  renderPlayers();
}

function removePlayer(id) {
  players = players.filter(p => p.id !== id);
  // if removed current player, move turn to next available
  if (currentPlayerId === id) {
    currentPlayerId = players.length ? players[0].id : null;
  }
  savePlayers();
  renderPlayers();
}

function changeScore(id, delta) {
  const p = players.find(x => x.id === id);
  if (!p) return;
  p.score += delta;
  // Any regular score change clears accumulated farkles
  if (delta !== 0) p.farkles = 0;

  // Winning threshold
  const THRESHOLD = 10000;

  // If the player reaches or exceeds the threshold record it.
  if (p.score >= THRESHOLD) {
    finalRoundReached.add(p.id);
    // If this is the first time anyone reached the threshold, start the final round
    if (!finalRound) {
      finalRound = true;
      finalRoundStarterId = p.id;
      // mark the starter as finished (they don't get another turn)
      finalRoundFinished.add(p.id);
      // show an informational modal announcing the final round
      showFinalRoundModal(p.name);
      // If only one player, end immediately
      if (players.length === 1) {
        endGame();
      }
    }
  }

  savePlayers();
  renderPlayers();
}

function getPlayerIndexById(id) {
  return players.findIndex(p => p.id === id);
}

function advanceTurn() {
  if (!players.length) {
    currentPlayerId = null;
    savePlayers();
    renderPlayers();
    return;
  }
  const idx = currentPlayerId ? getPlayerIndexById(currentPlayerId) : -1;
  const nextIdx = (idx + 1) % players.length;
  currentPlayerId = players[nextIdx].id;
  savePlayers();
  renderPlayers();

  // If the final round is active and we've returned to the starter, end the game.
  if (finalRound && currentPlayerId === finalRoundStarterId) {
    endGame();
  }
}

function applyScoreAction(id, delta) {
  changeScore(id, delta);
  // If final round is active, mark this player as having taken their final turn
  if (finalRound) finalRoundFinished.add(id);
  // advance turn after performing action
  advanceTurn();
}

function renderPlayers() {
  const tbody = document.getElementById('players-tbody');
  if (!tbody) return;
  tbody.innerHTML = '';
  if (players.length === 0) {
    const tr = document.createElement('tr');
    tr.innerHTML = '<td colspan="4" class="text-muted">No players yet</td>';
    tbody.appendChild(tr);
    return;
  }

  players.forEach((p, playerIndex) => {
    const isCurrent = currentPlayerId && currentPlayerId === p.id;
    const rowClass = isCurrent ? 'table-active' : '';
    const stripeClass = playerIndex % 2 === 0 ? 'stripe-light' : 'stripe-dark';
    
    // First row: Name (spans all columns)
    const tr1 = document.createElement('tr');
    // if the player has completed their final turn (or was the starter), mark row as finished
    const finishedClass = (typeof finalRoundFinished !== 'undefined' && finalRoundFinished.has && finalRoundFinished.has(p.id)) ? ' final-done' : '';
    tr1.className = rowClass + ' ' + stripeClass + finishedClass;
    tr1.innerHTML = `<td colspan="4" class="fw-bold">${escapeHtml(p.name)}</td>`;
    tbody.appendChild(tr1);
    
    // Second row: Farkle, Score, Actions
    const tr2 = document.createElement('tr');
    tr2.className = rowClass + ' ' + stripeClass + finishedClass;
    const farkleDisplay = 'F'.repeat(Math.max(0, Math.min(3, p.farkles || 0)));
    tr2.innerHTML = `
      <td class="text-center farkle-cell" aria-label="Farkles for ${escapeHtml(p.name)}">${escapeHtml(farkleDisplay)}</td>
      <td class="text-end">${p.score}</td>
      <td colspan="2" class="text-end">
        <div class="d-flex justify-content-end">
          <input ${isCurrent ? '' : 'disabled'} class="score-input form-control form-control-sm text-end" type="number" step="50" placeholder="+0" data-id="${p.id}" aria-label="Add score for ${escapeHtml(p.name)}">
          <button ${isCurrent ? 'disabled' : 'disabled'} class="btn btn-sm btn-primary score-add-btn ms-2" type="button" data-action="apply" data-id="${p.id}">Add</button>
          <button ${isCurrent ? '' : 'disabled'} class="btn btn-sm btn-outline-danger score-farkle-btn ms-2" type="button" data-action="farkle" data-id="${p.id}">Farkle</button>
        </div>
      </td>
    `;
    tbody.appendChild(tr2);
  });
}

function escapeHtml(str) {
  return String(str).replace(/[&<>"']/g, function (s) {
    return ({ '&': '&amp;', '<': '&lt;', '>': '&gt;', '"': '&quot;', "'": '&#39;' })[s];
  });
}

document.addEventListener('DOMContentLoaded', function () {
  showLandingScreen();
  loadGameList();
  renderSavedGamesList();

  // Close Game button logic
  document.getElementById('close-game-btn').addEventListener('click', function () {
    savePlayers();
    showLandingScreen();
  });

  // Winners modal "Close Game" button should behave the same as the top Close Game button.
  const winnersCloseBtn = document.getElementById('winners-close-btn');
  if (winnersCloseBtn) {
    winnersCloseBtn.addEventListener('click', function () {
      // Hide the winners modal if visible, persist state, and return to landing screen.
      try {
        const modalInst = bootstrap.Modal.getInstance(document.getElementById('winnersModal'));
        if (modalInst) modalInst.hide();
      } catch (e) {}
      savePlayers();
      showLandingScreen();
    });
  }

  // Landing screen buttons
  // Start New Game should prompt for a name via modal
  const newGameBtn = document.getElementById('start-new-game-btn');
  const newGameModalEl = document.getElementById('newGameModal');
  const newGameInput = document.getElementById('new-game-name-input');
  let newGameModal = null;
  if (newGameBtn && newGameModalEl && newGameInput) {
    newGameModal = new bootstrap.Modal(newGameModalEl);
    newGameBtn.addEventListener('click', function () {
      const defaultName = 'Game - ' + new Date().toLocaleString();
      newGameInput.value = defaultName;
      newGameModal.show();
      // focus the input after modal shown
      newGameModalEl.addEventListener('shown.bs.modal', function handler() {
        newGameInput.focus();
        newGameModalEl.removeEventListener('shown.bs.modal', handler);
      });
    });
  }
  document.getElementById('load-game-btn').addEventListener('click', function () {
    renderSavedGamesList();
    var modal = new bootstrap.Modal(document.getElementById('savedGamesModal'));
    modal.show();
  });

  // New Game modal confirm/cancel
  const confirmNewGameBtn = document.getElementById('confirm-new-game-btn');
  const cancelNewGameBtn = document.getElementById('cancel-new-game-btn');
  if (confirmNewGameBtn && newGameModal) {
    confirmNewGameBtn.addEventListener('click', function () {
      const name = newGameInput.value.trim();
      newGameModal.hide();
      startNewGame(name);
    });
  }
  if (cancelNewGameBtn && newGameModal) {
    cancelNewGameBtn.addEventListener('click', function () {
      newGameModal.hide();
    });
  }

  // Add player form
  const form = document.getElementById('add-player-form');
  const input = document.getElementById('player-name-input');
  if (form && input) {
    form.addEventListener('submit', function (e) {
      e.preventDefault();
      const name = input.value.trim();
      if (name) {
        addPlayer(name);
        input.value = '';
        input.focus();
      }
    });
  }

  // Player table event delegation
  const tbody = document.getElementById('players-tbody');
  if (tbody) {
    // Enable/disable Add button based on input value
    tbody.addEventListener('input', function (e) {
      const input = e.target.closest('input[data-id]');
      if (!input) return;
      const row = input.closest('tr');
      if (!row) return;
      const addBtn = row.querySelector('button[data-action="apply"]');
      if (addBtn) {
        const val = parseInt(input.value, 10) || 0;
        addBtn.disabled = val <= 0;
      }
    });
    // Apply on Enter key and ArrowUp/ArrowDown
    tbody.addEventListener('keydown', function (e) {
      if (e.key === 'Enter') {
        const input = e.target.closest('input[data-id]');
        if (!input) return;
        const id = input.getAttribute('data-id');
        if (currentPlayerId && currentPlayerId !== id) return;
        const val = parseInt(input.value, 10);
        if (!isNaN(val) && val !== 0) {
          applyScoreAction(id, val);
          input.value = '';
        }
        return;
      }
      if (e.key === 'ArrowUp' || e.key === 'ArrowDown') {
        const input = e.target.closest('input[data-id]');
        if (!input) return;
        const id = input.getAttribute('data-id');
        if (currentPlayerId && currentPlayerId !== id) return;
        const step = 50;
        const cur = parseInt(input.value, 10) || 0;
        const next = e.key === 'ArrowUp' ? cur + step : cur - step;
        input.value = next;
        e.preventDefault();
      }
    });
    tbody.addEventListener('blur', function (e) {
      if (ignoreNextBlur) { ignoreNextBlur = false; return; }
      const input = e.target.closest('input[data-id]');
      if (!input) return;
      const id = input.getAttribute('data-id');
      if (currentPlayerId && currentPlayerId !== id) return;
      const val = parseInt(input.value, 10);
      if (!isNaN(val) && val !== 0) {
        applyScoreAction(id, val);
        input.value = '';
      }
    }, true);
    tbody.addEventListener('pointerdown', function (e) {
      const btn = e.target.closest('button[data-action]');
      if (btn) {
        ignoreNextBlur = true;
      }
    });
    tbody.addEventListener('click', function (e) {
      const btn = e.target.closest('button[data-action]');
      if (!btn) return;
      const action = btn.getAttribute('data-action');
      const id = btn.getAttribute('data-id');
      if (action === 'apply') {
        if (currentPlayerId && currentPlayerId !== id) return;
        const input = tbody.querySelector(`input[data-id="${id}"]`);
        if (!input) return;
        const val = parseInt(input.value, 10);
        if (!isNaN(val) && val !== 0) {
          applyScoreAction(id, val);
          input.value = '';
        }
      } else if (action === 'farkle') {
        if (currentPlayerId && currentPlayerId !== id) return;
        const p = players.find(x => x.id === id);
        if (!p) return;
        p.farkles = (p.farkles || 0) + 1;
        if (p.farkles >= 3) {
          p.score = p.score - 1000;
          p.farkles = 0;
        }
        // mark player's final turn as taken if we're in the final round
        if (finalRound) finalRoundFinished.add(p.id);
        savePlayers();
        advanceTurn();
      }
    });
  }

  // Saved games modal logic
  document.getElementById('saved-games-list').addEventListener('click', function (e) {
    const btn = e.target.closest('button[data-action]');
    if (!btn) return;
    const action = btn.getAttribute('data-action');
    const gid = btn.getAttribute('data-id');
    if (action === 'load') {
      loadPlayers(gid);
      // Hide the saved-games modal if open
      var savedModalInst = bootstrap.Modal.getInstance(document.getElementById('savedGamesModal'));
      if (savedModalInst) savedModalInst.hide();

      // If the loaded game has already ended, show the score sheet in the background
      // and then display the persistent Winners modal on top.
      if (typeof gameEnded !== 'undefined' && gameEnded) {
        // Show the game UI (score sheet) in the background so the user can view final scores.
        showGameUI();

        // Disable interactive inputs in the score sheet because the game is finished.
        // Disable inputs and action buttons inside the players table.
        try {
          const controls = document.querySelectorAll('#players-tbody input, #players-tbody button, #add-player-form input, #add-player-form button');
          controls.forEach(el => el.disabled = true);
        } catch (e) {
          // ignore any errors
        }

        // Also hide the landing screen if visible
        const landingEl = document.getElementById('landing-screen');
        if (landingEl) {
          landingEl.style.display = 'none';
          landingEl.hidden = true;
          landingEl.setAttribute('aria-hidden', 'true');
        }

        // Show the winners modal using the saved snapshot
        showWinnersModal(savedWinners);
      } else {
        // Normal (not-ended) game load — show the interactive game UI
        showGameUI();
      }
    } else if (action === 'delete') {
      deleteTargetGameId = gid;
      document.getElementById('delete-confirm').style.display = '';
      document.getElementById('saved-games-list').style.display = 'none';
    }
  });
  document.getElementById('confirm-delete-btn').addEventListener('click', function () {
    if (!deleteTargetGameId) return;
    localStorage.removeItem(deleteTargetGameId);
    savedGames = savedGames.filter(g => g.id !== deleteTargetGameId);
    saveGameList();
    deleteTargetGameId = null;
    document.getElementById('delete-confirm').style.display = 'none';
    document.getElementById('saved-games-list').style.display = '';
    renderSavedGamesList();
  });
  document.getElementById('cancel-delete-btn').addEventListener('click', function () {
    deleteTargetGameId = null;
    document.getElementById('delete-confirm').style.display = 'none';
    document.getElementById('saved-games-list').style.display = '';
  });

console.log('FSK initialized');
});

// End-game handling ---------------------------------------------------------

function endGame() {
  // Determine winner(s) by highest score after final round ends.
  if (!players.length) return;

  const maxScore = Math.max(...players.map(p => p.score));
  const winners = players.filter(p => p.score === maxScore);

  // Stop further actions by clearing current player
  currentPlayerId = null;

  // Persist the fact the game ended and store a snapshot of winners for future loads
  gameEnded = true;
  savedWinners = winners.map(w => ({ id: w.id, name: w.name, score: w.score }));

  // Update savedGames metadata so the Load list can show ended status immediately
  try {
    const entry = savedGames.find(s => s.id === gameId);
    if (entry) {
      entry.ended = true;
      entry.winnerNames = savedWinners.map(w => w.name);
      saveGameList();
    }
  } catch (e) {
    // ignore
  }

  savePlayers();
  renderPlayers();

  // Show winners modal (persistent)
  showWinnersModal(winners);

  // Reset final round meta so the game can later be reused or restarted
  finalRound = false;
  finalRoundStarterId = null;
  finalRoundReached = new Set();

  // Hide the inline status banner — the persistent winners modal shows the result instead.
  const statusEl = document.getElementById('status');
  if (statusEl) {
    statusEl.style.display = 'none';
  }
}

function showWinnersModal(winners) {
  const modalEl = document.getElementById('winnersModal');
  if (!modalEl) return;

  // Auto-save winners state immediately so the saved-games list and localStorage
  // reliably reflect that this game ended, even if the user closes the page.
  try {
    gameEnded = true;
    savedWinners = winners.map(w => ({ id: w.id, name: w.name, score: w.score }));
    // Update savedGames metadata so the list can show ended status immediately
    const entry = savedGames.find(s => s.id === gameId);
    if (entry) {
      entry.ended = true;
      entry.winnerNames = savedWinners.map(w => w.name);
      saveGameList();
    }
    // Persist per-game data as well
    savePlayers();
  } catch (e) {
    // ignore persistence errors
  }

  const listEl = modalEl.querySelector('.winners-list');
  const titleEl = modalEl.querySelector('.winners-title');
  if (titleEl) {
    titleEl.textContent = winners.length === 1 ? 'Winner' : 'Winners';
  }
  if (listEl) {
    listEl.innerHTML = '';
    if (!winners.length) {
      listEl.innerHTML = '<li class="list-group-item">No winners</li>';
    } else {
      winners.forEach(w => {
        const li = document.createElement('li');
        li.className = 'list-group-item d-flex justify-content-between align-items-center';
        li.innerHTML = `<span>${escapeHtml(w.name)}</span><span class="badge bg-primary rounded-pill">${w.score}</span>`;
        listEl.appendChild(li);
      });
    }
  }

  // Add class to trigger simple CSS fireworks animation
  modalEl.classList.add('show-fireworks');
  const modal = new bootstrap.Modal(modalEl, { backdrop: 'static', keyboard: false });
  modal.show();

  // Clean up fireworks class when modal is hidden (modal is persistent so this may not run)
  modalEl.addEventListener('hidden.bs.modal', function handler() {
    modalEl.classList.remove('show-fireworks');
    modalEl.removeEventListener('hidden.bs.modal', handler);
  });
}

function showFinalRoundModal(starterName) {
  const modalEl = document.getElementById('finalRoundModal');
  if (!modalEl) return;
  const titleEl = modalEl.querySelector('.finalround-title');
  const bodyEl = modalEl.querySelector('.finalround-body');
  if (titleEl) titleEl.textContent = 'Final Round';
  if (bodyEl) bodyEl.textContent = `${starterName} reached 10000. Other players get one final turn.`;
  const modal = new bootstrap.Modal(modalEl);
  modal.show();
}

function renderSavedGamesList() {
  loadGameList();
  const deleteConfirm = document.getElementById('delete-confirm');
  const listEl = document.getElementById('saved-games-list');
  if (deleteConfirm) deleteConfirm.style.display = 'none';
  if (listEl) listEl.style.display = '';
  const ul = listEl;
  if (!ul) return;
  ul.innerHTML = '';
  if (!savedGames.length) {
    ul.innerHTML = '<li class="list-group-item text-muted">No saved games found.</li>';
    return;
  }
  savedGames.forEach(g => {
    const li = document.createElement('li');
    li.className = 'list-group-item d-flex justify-content-between align-items-center';
    // detect whether the saved game is already ended and show a marker in the list
    let endedLabel = '';
    try {
      const raw = localStorage.getItem(g.id);
      const data = raw ? JSON.parse(raw) : null;
      if (data) {
        // Priority 1: explicit saved winners snapshot
        if (data.savedWinners && data.savedWinners.length) {
          const names = data.savedWinners.map(w => escapeHtml(w.name)).join(', ');
          endedLabel = ` <span class="text-danger small">(ENDED)</span>`;
        // Priority 2: explicit gameEnded flag
        } else if (data.gameEnded) {
          endedLabel = ' <span class="text-danger small">(ENDED)</span>';
        // Priority 3: infer from player scores in older saves (player reached threshold)
        } else if (Array.isArray(data.players)) {
          const THRESHOLD = 10000;
          const winners = data.players.filter(p => p && typeof p.score === 'number' && p.score >= THRESHOLD);
          if (winners.length) {
            const names = winners.map(w => escapeHtml(w.name)).join(', ');
            endedLabel = ` <span class="text-danger small">(ENDED)</span>`;
          }
        }
      }
    } catch (e) {
      endedLabel = '';
    }
    li.innerHTML = `<span>${escapeHtml(g.name)}${endedLabel}</span> <span><button class="btn btn-sm btn-primary me-2" data-action="load" data-id="${g.id}">Load</button><button class="btn btn-sm btn-danger" data-action="delete" data-id="${g.id}">Delete</button></span>`;
    ul.appendChild(li);
  });
}
