<?php
namespace BuroMikado\Modules\Shortcodes\Button;

use BuroMikado\Modules\Shortcodes\Lib\ShortcodeInterface;


/**
 * Class Button that represents button shortcode
 * @package BuroMikado\Modules\Shortcodes\Button
 */
class Button implements ShortcodeInterface {
	/**
	 * @var string
	 */
	private $base;

	/**
	 * Sets base attribute and registers shortcode with Visual Composer
	 */
	public function __construct() {
		$this->base = 'mkd_button';

		add_action('vc_before_init', array($this, 'vcMap'));
	}

	/**
	 * Returns base attribute
	 * @return string
	 */
	public function getBase() {
		return $this->base;
	}

	/**
	 * Maps shortcode to Visual Composer
	 */
	public function vcMap() {
		vc_map(array(
			'name'                      => esc_html__('Button', 'buro'),
			'base'                      => $this->base,
			'category'                  => esc_html__('by MIKADO', 'buro'),
			'icon'                      => 'icon-wpb-button extended-custom-icon',
			'allowed_container_element' => 'vc_row',
			'params'                    => array_merge(
				array(
					array(
						'type'        => 'dropdown',
						'heading'     => esc_html__('Size', 'buro'),
						'param_name'  => 'size',
						'value'       => array(
							esc_html__('Default','buro') => '',
							esc_html__('Small', 'buro')  => 'small',
							esc_html__('Medium', 'buro') => 'medium',
							esc_html__('Large', 'buro')  => 'large',
							esc_html__('Extra Large', 'buro') => 'huge',
							esc_html__('Full Width', 'buro')  => 'huge-full-width'
						)
					),
					array(
						'type'        => 'dropdown',
						'heading'     => esc_html__('Type', 'buro'),
						'param_name'  => 'type',
						'value'       => array(
							esc_html__('Default', 'buro') => '',
							esc_html__('Outline', 'buro') => 'outline',
							esc_html__('Solid', 'buro') => 'solid',
                            esc_html__('Transparent', 'buro') => 'transparent',
                            esc_html__('Circled', 'buro') => 'circled'
						),
						'save_always' => true,
					),
					array(
						'type'        => 'textfield',
						'heading'     => esc_html__('Text', 'buro'),
						'param_name'  => 'text',
                        'dependency'  => array('element' => 'type', 'value' => array('', 'outline', 'solid', 'transparent'))
					),
					array(
						'type'        => 'textfield',
						'heading'     => esc_html__('Link', 'buro'),
						'param_name'  => 'link',
					),
					array(
						'type'        => 'dropdown',
						'heading'     => esc_html__('Link Target', 'buro'),
						'param_name'  => 'target',
						'value'       => array(
							esc_html__('Self', 'buro')  => '_self',
							esc_html__('Blank', 'buro') => '_blank'
						),
						'save_always' => true,
					),
					array(
						'type'        => 'textfield',
						'heading'     => esc_html__('Custom CSS class', 'buro'),
						'param_name'  => 'custom_class',
					)
				),
				buro_mikado_icon_collections()->getVCParamsArray(array(), '', true),
				array(
					array(
						'type'        => 'colorpicker',
						'heading'     => esc_html__('Color', 'buro'),
						'param_name'  => 'color',
						'group'       => esc_html__('Design Options','buro'),
					),
					array(
						'type'        => 'colorpicker',
						'heading'     => esc_html__('Hover Color', 'buro'),
						'param_name'  => 'hover_color',
						'group'       => esc_html__('Design Options','buro'),
					),
					array(
						'type'        => 'colorpicker',
						'heading'     => esc_html__('Background Color', 'buro'),
						'param_name'  => 'background_color',
						'group'       => esc_html__('Design Options','buro'),
						'dependency'  => array('element' => 'type', 'value' => array('solid',''))
					),
					array(
						'type'        => 'colorpicker',
						'heading'     => esc_html__('Hover Background Color', 'buro'),
						'param_name'  => 'hover_background_color',
						'group'       => esc_html__('Design Options','buro'),
						'dependency'  => array('element' => 'type', 'value' => array('solid','outline',''))
					),
					array(
						'type'        => 'colorpicker',
						'heading'     => esc_html__('Border Color', 'buro'),
						'param_name'  => 'border_color',
						'group'       => esc_html__('Design Options','buro'),
						'dependency'  => array('element' => 'type', 'value' => array('solid','outline',''))
					),
					array(
						'type'        => 'colorpicker',
						'heading'     => esc_html__('Hover Border Color', 'buro'),
						'param_name'  => 'hover_border_color',
						'group'       => esc_html__('Design Options','buro'),
						'dependency'  => array('element' => 'type', 'value' => array('solid','outline',''))
					),
					array(
						'type'        => 'textfield',
						'heading'     => esc_html__('Font Size (px)', 'buro'),
						'param_name'  => 'font_size',
						'group'       => esc_html__('Design Options','buro'),
					),
					array(
						'type'        => 'dropdown',
						'heading'     => esc_html__('Font Weight', 'buro'),
						'param_name'  => 'font_weight',
						'value'       => array_flip(buro_mikado_get_font_weight_array(true)),
						'group'       => esc_html__('Design Options','buro'),
					),
					array(
						'type'        => 'textfield',
						'heading'     => esc_html__('Margin', 'buro'),
						'param_name'  => 'margin',
						'description' => esc_html__('Insert margin in format: 0px 0px 1px 0px', 'buro'),
						'group'       => esc_html__('Design Options','buro'),
					),
					array(
						'type'        => 'colorpicker',
						'heading'     => esc_html__('Icon Color', 'buro'),
						'param_name'  => 'icon_color',
						'group'       => esc_html__('Design Options','buro'),
					),
					array(
						'type'        => 'colorpicker',
						'heading'     => esc_html__('Icon Background Color', 'buro'),
						'param_name'  => 'icon_background_color',
						'group'       => esc_html__('Design Options','buro'),
						'dependency'  => array('element' => 'type', 'value' => array('solid',''))
					)
				)
			) //close array_merge
		));
	}

	/**
	 * Renders HTML for button shortcode
	 *
	 * @param array $atts
	 * @param null $content
	 *
	 * @return string
	 */
	public function render($atts, $content = null) {
		$default_atts = array(
			'size'                   => 'medium',
			'type'                   => '',
			'text'                   => '',
			'link'                   => '',
			'target'                 => '',
			'color'                  => '',
			'background_color'       => '',
			'border_color'           => '',
			'hover_color'            => '',
			'hover_background_color' => '',
			'hover_border_color'     => '',
			'font_size'              => '',
			'font_weight'            => '',
			'icon_color'			 => '',
			'icon_background_color'	 => '',
			'margin'                 => '',
			'custom_class'           => '',
			'html_type'              => 'anchor',
			'input_name'             => '',
			'custom_attrs'           => array()
		);

		$default_atts = array_merge($default_atts, buro_mikado_icon_collections()->getShortcodeParams());
		$params       = shortcode_atts($default_atts, $atts);

        if($params['html_type'] !== 'input') {
        	$icon_attributes = array();

            $iconPackName   = buro_mikado_icon_collections()->getIconCollectionParamNameByKey($params['icon_pack']);
            $params['icon'] = $iconPackName ? $params[$iconPackName] : '';

			$icon_attributes['class'] = 'mkd-btn-icon-holder';
        }

		$params['type'] = !empty($params['type']) ? $params['type'] : 'solid';


		$params['link']   = !empty($params['link']) ? $params['link'] : '#';
		$params['target'] = !empty($params['target']) ? $params['target'] : '_self';

		//prepare params for template
		$params['button_classes']      = $this->getButtonClasses($params);
		$params['button_custom_attrs'] = !empty($params['custom_attrs']) ? $params['custom_attrs'] : array();
		$params['button_styles']       = $this->getButtonStyles($params);
        $params['button_data']         = $this->getButtonDataAttr($params);
		$params['icon_styles']       = $this->getIconStyle($params);

		return mkd_core_get_module_shortcode_template_part('templates/'.$params['html_type'], 'button', '', $params);
	}

	/**
	 * Returns array of button styles
	 *
	 * @param $params
	 *
	 * @return array
	 */
	private function getButtonStyles($params) {
		$styles = array();

		if(!empty($params['color'])) {
			$styles[] = 'color: '.$params['color'];
		}

		if(!empty($params['background_color']) && $params['type'] !== 'outline') {
			$styles[] = 'background-color: '.$params['background_color'];
		}

		if(!empty($params['border_color'])) {
			$styles[] = 'border: 3px solid '.$params['border_color'];
		}

		if(!empty($params['font_size'])) {
			$styles[] = 'font-size: '.buro_mikado_filter_px($params['font_size']).'px';
		}

		if(!empty($params['font_weight'])) {
			$styles[] = 'font-weight: '.$params['font_weight'];
		}

		if(!empty($params['margin'])) {
			$styles[] = 'margin: '.$params['margin'];
		}

		return $styles;
	}

    /**
     *
     * Returns array of button data attr
     *
     * @param $params
     *
     * @return array
     */
    private function getButtonDataAttr($params) {
        $data = array();

        if(!empty($params['hover_background_color'])) {
            $data['data-hover-bg-color'] = $params['hover_background_color'];
        }

        if(!empty($params['hover_color'])) {
            $data['data-hover-color'] = $params['hover_color'];
        }

        if(!empty($params['hover_border_color'])) {
            $data['data-hover-border-color'] = $params['hover_border_color'];
        }

        return $data;
    }

	/**
	 * Returns array of HTML classes for button
	 *
	 * @param $params
	 *
	 * @return array
	 */
	private function getButtonClasses($params) {
		$buttonClasses = array(
			'mkd-btn',
			'mkd-btn-'.$params['size'],
			'mkd-btn-'.$params['type']
		);

		if(!empty($params['icon'])) {
			$buttonClasses[] = 'mkd-btn-icon';
		}

		if (empty($params['hover_background_color'])){
			$buttonClasses[] = 'mkd-btn-bckg-hover';
		}

        if(!empty($params['hover_background_color'])) {
            $buttonClasses[] = 'mkd-btn-custom-hover-bg';
        }

        if(!empty($params['hover_border_color'])) {
            $buttonClasses[] = 'mkd-btn-custom-border-hover';
        }

        if(!empty($params['hover_color'])) {
            $buttonClasses[] = 'mkd-btn-custom-hover-color';
        }

		if(!empty($params['custom_class'])) {
			$buttonClasses[] = $params['custom_class'];
		}

		return $buttonClasses;
	}

	/**
	 * Returns icon style
	 *
	 * @param $params
	 *
	 * @return string
	 */
	private function getIconStyle($params) {
		$style = '';

		if(!empty($params['icon_background_color'])) {
			$style .= 'background-color: '.$params['icon_background_color'].';';
		}

		if(!empty($params['icon_color'])) {
			$style .= 'color: '.$params['icon_color'];
		}

		return $style;
	}
}