<?php
namespace BuroMikado\Modules\Shortcodes\ImageGallery;

use BuroMikado\Modules\Shortcodes\Lib\ShortcodeInterface;

class ImageGallery implements ShortcodeInterface{

	private $base;

	/**
	 * Image Gallery constructor.
	 */
	public function __construct() {
		$this->base = 'mkd_image_gallery';

		add_action('vc_before_init', array($this, 'vcMap'));
	}

	/**
	 * Returns base for shortcode
	 * @return string
	 */
	public function getBase() {
		return $this->base;
	}

	/**
	 * Maps shortcode to Visual Composer. Hooked on vc_before_init
	 *
	 */
	public function vcMap() {

		vc_map(array(
			'name'                      => esc_html__('Image Gallery', 'buro'),
			'base'                      => $this->getBase(),
			'category'                  => esc_html__('by MIKADO', 'buro'),
			'icon'                      => 'icon-wpb-image-gallery extended-custom-icon',
			'allowed_container_element' => 'vc_row',
			'params'                    => array(
				array(
					'type'			=> 'attach_images',
					'heading'		=> esc_html__('Images', 'buro'),
					'param_name'	=> 'images',
					'description'	=> esc_html__('Select images from media library', 'buro')
				),
				array(
					'type'        => 'dropdown',
					'heading'     => esc_html__('Gallery Type', 'buro'),
					'admin_label' => true,
					'param_name'  => 'type',
					'value'       => array(
						esc_html__('Slider', 'buro')		=> 'slider',
						esc_html__('Carousel', 'buro')	=> 'carousel',
						esc_html__('Image Grid', 'buro')	=> 'image_grid',
						esc_html__('Masonry', 'buro')	=> 'masonry',
					),
					'description' => esc_html__('Select gallery type', 'buro'),
					'save_always' => true
				),
				array(
					'type'			=> 'textfield',
					'heading'		=> esc_html__('Image Size', 'buro'),
					'param_name'	=> 'image_size',
					'description'	=> esc_html__('Enter image size. Example: thumbnail, medium, large, full or other sizes defined by current theme. Alternatively enter image size in pixels: 200x100 (Width x Height). Leave empty to use "thumbnail" size', 'buro'),
					'dependency'	=> array('element' => 'type', 'value' => array('slider','image_grid','carousel'))
				),
				array(
					'type'			=> 'dropdown',
					'heading'		=> esc_html__('Slide duration', 'buro'),
					'admin_label'	=> true,
					'param_name'	=> 'autoplay',
					'value'			=> array(
						'3'			=> '3',
						'5'			=> '5',
						'10'		=> '10',
						esc_html__('Disable','buro')	=> 'disable'
					),
					'save_always'	=> true,
					'dependency'	=> array('element'	=> 'type','value'=> array('slider','carousel')),
					'description' => esc_html__('Auto rotate slides each X seconds', 'buro'),
				),
				array(
					'type'			=> 'dropdown',
					'heading'		=> esc_html__('Slide Animation', 'buro'),
					'admin_label'	=> true,
					'param_name'	=> 'slide_animation',
					'value'			=> array(
						esc_html__('Slide', 'buro')	=> 'slide',
						esc_html__('Fade','buro')	=> 'fade'
					),
					'save_always'	=> true,
					'dependency'	=> array(
						'element'	=> 'type',
						'value'		=> array(
							'slider'
						)
					)
				),
				array(
					'type'			=> 'dropdown',
					'heading'		=> esc_html__('Column Number', 'buro'),
					'param_name'	=> 'column_number',
					'value'			=> array(2, 3, 4, 5),
					'save_always'	=> true,
					'dependency'	=> array(
						'element' 	=> 'type',
						'value'		=> array(
							'image_grid'
						)
					),
				),
                array(
                    'type'           => 'dropdown',
                    'heading'        => esc_html__('Spaces between images','buro'),
                    'param_name'     => 'images_space',
                    'value'          => array(
                        esc_html__('Yes','buro') => 'yes',
                        esc_html__('No','buro') => 'no',
                    ),
                    'dependency'     => array(
                        'element'    => 'type',
                        'value'      => array(
                            'image_grid',
                            'masonry'
                        )
                    )
                ),
				array(
					'type'			=> 'dropdown',
					'heading'		=> esc_html__('Open PrettyPhoto on click', 'buro'),
					'param_name'	=> 'pretty_photo',
					'value'			=> array(
						esc_html__('No', 'buro')		=> 'no',
						esc_html__('Yes', 'buro')	=> 'yes'
					),
					'save_always'	=> true,
				),
				array(
					'type' => 'dropdown',
					'heading' => esc_html__('Grayscale Images', 'buro'),
					'param_name' => 'grayscale',
					'value' => array(
						esc_html__('No', 'buro') => 'no',
						esc_html__('Yes', 'buro') => 'yes'
					),
					'save_always'	=> true,
					'dependency'	=> array(
						'element'	=> 'type',
						'value'		=> array(
							'image_grid'
						)
					)
				),
				array(
					'type'			=> 'dropdown',
					'heading'		=> esc_html__('Show Navigation Arrows', 'buro'),
					'param_name' 	=> 'navigation',
					'value' 		=> array(
						esc_html__('Yes', 'buro') => 'yes',
						esc_html__('No', 'buro')	=> 'no'
					),
					'dependency' 	=> array(
						'element' => 'type',
						'value' => array(
							'slider',
							'carousel'
						)
					),
					'save_always'	=> true
				),
				array(
					'type'			=> 'dropdown',
					'heading'		=> esc_html__('Show Pagination', 'buro'),
					'param_name' 	=> 'pagination',
					'value' 		=> array(
						esc_html__('Yes', 'buro') => 'yes',
						esc_html__('No', 'buro')	=> 'no'
					),
					'save_always'	=> true,
					'dependency'	=> array(
						'element' => 'type',
						'value' => array(
							'slider',
							'carousel'
						)
					)
				)
			)
		));

	}

	/**
	 * Renders shortcodes HTML
	 *
	 * @param $atts array of shortcode params
	 * @param $content string shortcode content
	 * @return string
	 */
	public function render($atts, $content = null) {

		$args = array(
			'images'			=> '',
			'image_size'		=> 'thumbnail',
			'type'				=> 'slider',
			'autoplay'			=> '5000',
			'slide_animation'	=> 'slide',
			'images_space'		=> 'yes',
			'pretty_photo'		=> '',
			'column_number'		=> '',
			'grayscale'			=> '',
			'navigation'		=> 'yes',
			'pagination'		=> 'yes'
		);

		$params = shortcode_atts($args, $atts);
		$params['slider_data'] = $this->getSliderData($params);
		$params['image_size'] = $this->getImageSize($params['image_size']);
		$params['images'] = $this->getGalleryImages($params);
		$params['pretty_photo'] = ($params['pretty_photo'] == 'yes') ? true : false;
		$params['columns'] = 'mkd-gallery-columns-' . $params['column_number'];
		$params['gallery_classes'] = $this->getGalleryClasses($params);
		$params['slider_classes'] = $this->getSliderClasses($params);

		if ($params['type'] == 'image_grid') {
			$template = 'gallery-grid';
		} elseif ($params['type'] == 'slider' || $params['type'] == 'carousel') {
			$template = 'gallery-slider';
		} elseif ($params['type'] == 'masonry') {
			$template = 'gallery-masonry';
		}

		$html = mkd_core_get_module_shortcode_template_part('templates/' . $template, 'imagegallery', '', $params);

		return $html;

	}

	/**
	 * Return images for gallery
	 *
	 * @param $params
	 * @return array
	 */
	private function getGalleryImages($params) {
		$image_ids = array();
		$images = array();
		$i = 0;

		if ($params['images'] !== '') {
			$image_ids = explode(',', $params['images']);
		}

		foreach ($image_ids as $id) {

			$image['image_id'] = $id;
			$image['class'] = '';
			if ($params['type'] == 'masonry') {
		        $size = get_post_meta($id,'_gallery_masonry_image_size', true);
		        $size = ($size)?$size:'mkd-default-masonry-item';
		        switch($size){
			        case 'mkd-large-height-masonry-item' :
				        $img_size = 'buro_mikado_large_height';
				        $image['class'] = 'mkd-size-portrait';
				        break;
			        case 'mkd-large-width-masonry-item' :
				        $img_size = 'buro_mikado_large_width';
				        $image['class'] = 'mkd-size-landscape';
				        break;
			        case 'mkd-large-width-height-masonry-item' :
				        $img_size = 'buro_mikado_large_width_height';
				        $image['class'] = 'mkd-size-big-square';
				        break;
			        default:
				        $img_size = 'buro_mikado_square';
				        $image['class'] = 'mkd-size-square';
				        break;
		        }
			}
			else{
				$img_size = 'full';
			}
			$image_original = wp_get_attachment_image_src($id, $img_size);
			$image['masonry_size'] = $img_size;
			$image['url'] = $image_original[0];
			$image['title'] = get_the_title($id);
			$image['link'] = get_post_meta($id,'_attachment_image_custom_link', true);
			$image['link_target'] = get_post_meta($id,'_attachment_image_link_target', true);

			if ($image['link_target'] == ''){
				$image['link_target'] = '_self';
			}

			$images[$i] = $image;
			$i++;
		}

		return $images;

	}

	/**
	 * Return image size or custom image size array
	 *
	 * @param $image_size
	 * @return array
	 */
	private function getImageSize($image_size) {

		$image_size = trim($image_size);
		//Find digits
		preg_match_all( '/\d+/', $image_size, $matches );
		if(in_array( $image_size, array('thumbnail', 'thumb', 'medium', 'large', 'full'))) {
			return $image_size;
		} elseif(!empty($matches[0])) {
			return array(
					$matches[0][0],
					$matches[0][1]
			);
		} else {
			return 'thumbnail';
		}
	}

	/**
	 * Return all configuration data for slider
	 *
	 * @param $params
	 * @return array
	 */
	private function getSliderData($params) {

		$slider_data = array();

		$slider_data['data-autoplay'] = ($params['autoplay'] !== '') ? $params['autoplay'] : '';
		$slider_data['data-animation'] = ($params['slide_animation'] !== '') ? $params['slide_animation'] : '';
		$slider_data['data-navigation'] = ($params['navigation'] !== '') ? $params['navigation'] : '';
		$slider_data['data-pagination'] = ($params['pagination'] !== '') ? $params['pagination'] : '';

		return $slider_data;

	}

    /**
     * Generates css classes for Gallery
     *
     * @param $params
     *
     * @return array
     */
    private function getGalleryClasses($params){

        $gallery_classes = array();

        if ($params['images_space'] == 'no'){
            $gallery_classes[] = 'mkd-gallery-no-space';
        }
        else{
            $gallery_classes[] = 'mkd-gallery-with-space';
        }

        if ($params['grayscale'] == 'yes' && $params['type'] == 'image_grid'){
            $gallery_classes[] = 'mkd-grayscale';
        }

        return implode(' ', $gallery_classes);
    }

    /**
     * Generates css classes for slider and carousel
     *
     * @param $params
     *
     * @return array
     */
    private function getSliderClasses($params){

        $classes = array();

        if ($params['type'] == 'carousel') {
        	$classes[] = 'mkd-gallery-image-carousel';
        }
        elseif ($params['type'] == 'slider'){
        	$classes[] = 'mkd-gallery-image-slider';
        }

        return implode(' ', $classes);
    }
}