<?php
namespace BuroMikado\Modules\Header\Types;

use BuroMikado\Modules\Header\Lib\HeaderType;

/**
 * Class that represents Header Centered layout and option
 *
 * Class HeaderCentered
 */
class HeaderCentered extends HeaderType {
    protected $heightOfTransparency;
    protected $heightOfCompleteTransparency;
    protected $headerHeight;
    protected $mobileHeaderHeight;

    /**
     * Sets slug property which is the same as value of option in DB
     */
    public function __construct() {
        $this->slug = 'header-centered';

        if(!is_admin()) {

            $menuAreaHeight       = buro_mikado_filter_px(buro_mikado_options()->getOptionValue('menu_area_height_header_centered'));
            $this->menuAreaHeight = $menuAreaHeight !== '' ? (int)$menuAreaHeight : 90;

            $mobileHeaderHeight       = buro_mikado_filter_px(buro_mikado_options()->getOptionValue('mobile_header_height'));
            $this->mobileHeaderHeight = $mobileHeaderHeight !== '' ? (int)$mobileHeaderHeight : 100;

            add_action('wp', array($this, 'setHeaderHeightProps'));

            add_filter('buro_mikado_js_global_variables', array($this, 'getGlobalJSVariables'));
            add_filter('buro_mikado_per_page_js_vars', array($this, 'getPerPageJSVariables'));

        }
    }

    /**
     * Loads template file for this header type
     *
     * @param array $parameters associative array of variables that needs to passed to template
     */
    public function loadTemplate($parameters = array()) {

        $parameters['menu_area_in_grid'] = buro_mikado_options()->getOptionValue('menu_area_in_grid_header_centered') == 'yes' ? true : false;

        $parameters = apply_filters('buro_mikado_header_centered_parameters', $parameters);

        buro_mikado_get_module_template_part('templates/types/'.$this->slug, $this->moduleName, '', $parameters);
    }

    /**
     * Sets header height properties after WP object is set up
     */
    public function setHeaderHeightProps(){
        $this->heightOfTransparency         = $this->calculateHeightOfTransparency();
        $this->heightOfCompleteTransparency = $this->calculateHeightOfCompleteTransparency();
        $this->headerHeight                 = $this->calculateHeaderHeight();
        $this->mobileHeaderHeight           = $this->calculateMobileHeaderHeight();
    }

    /**
     * Returns total height of transparent parts of header
     *
     * @return int
     */
    public function calculateHeightOfTransparency() {
        $id = buro_mikado_get_page_id();
        $transparencyHeight = 0;

        if(get_post_meta($id, 'mkd_menu_area_background_color_header_centered_meta', true) !== ''){
            $menuAreaTransparent = get_post_meta($id, 'mkd_menu_area_background_color_header_centered_meta', true) !== '' &&
                                   get_post_meta($id, 'mkd_menu_area_background_transparency_header_centered_meta', true) !== '1';
        } else {
            $menuAreaTransparent = buro_mikado_options()->getOptionValue('menu_area_background_color_header_centered') !== '' &&
                                   buro_mikado_options()->getOptionValue('menu_area_background_transparency_header_centered') !== '1';
        }


        $sliderExists = get_post_meta($id, 'mkd_page_slider_meta', true) !== '';

        if($sliderExists){
            $menuAreaTransparent = true;
        }

        if($menuAreaTransparent) {
            $transparencyHeight = $this->menuAreaHeight;

            if(($sliderExists && buro_mikado_is_top_bar_enabled())
               || buro_mikado_is_top_bar_enabled() && buro_mikado_is_top_bar_transparent()) {
                $transparencyHeight += buro_mikado_get_top_bar_height();
            }
        }

        return $transparencyHeight;
    }

    /**
     * Returns height of completely transparent header parts
     *
     * @return int
     */
    public function calculateHeightOfCompleteTransparency() {
        $id = buro_mikado_get_page_id();
        $transparencyHeight = 0;

        if(get_post_meta($id, 'mkd_menu_area_background_color_header_centered_meta', true) !== ''){
            $menuAreaTransparent = get_post_meta($id, 'mkd_menu_area_background_color_header_centered_meta', true) !== '' &&
                                   get_post_meta($id, 'mkd_menu_area_background_transparency_header_centered_meta', true) === '0';
        } else {
            $menuAreaTransparent = buro_mikado_options()->getOptionValue('menu_area_background_color_header_centered') !== '' &&
                                   buro_mikado_options()->getOptionValue('menu_area_background_transparency_header_centered') === '0';
        }

        if($menuAreaTransparent) {
            $transparencyHeight = $this->menuAreaHeight;
        }

        return $transparencyHeight;
    }


    /**
     * Returns total height of header
     *
     * @return int|string
     */
    public function calculateHeaderHeight() {
        $headerHeight = $this->menuAreaHeight;
        if(buro_mikado_is_top_bar_enabled()) {
            $headerHeight += buro_mikado_get_top_bar_height();
        }

        return $headerHeight;
    }

    /**
     * Returns total height of mobile header
     *
     * @return int|string
     */
    public function calculateMobileHeaderHeight() {
        $mobileHeaderHeight = $this->mobileHeaderHeight;

        return $mobileHeaderHeight;
    }

    /**
     * Returns global js variables of header
     *
     * @param $globalVariables
     * @return int|string
     */
    public function getGlobalJSVariables($globalVariables) {
        $globalVariables['mkdLogoAreaHeight'] = 0;
        $globalVariables['mkdMenuAreaHeight'] = $this->headerHeight;
        $globalVariables['mkdMobileHeaderHeight'] = $this->mobileHeaderHeight;

        return $globalVariables;
    }

    /**
     * Returns per page js variables of header
     *
     * @param $perPageVars
     * @return int|string
     */
    public function getPerPageJSVariables($perPageVars) {
        //calculate transparency height only if header has no sticky behaviour
        if(!in_array(buro_mikado_options()->getOptionValue('header_behaviour'), array('sticky-header-on-scroll-up','sticky-header-on-scroll-down-up'))) {
            $perPageVars['mkdHeaderTransparencyHeight'] = $this->headerHeight - (buro_mikado_get_top_bar_height() + $this->heightOfCompleteTransparency);
        }else{
            $perPageVars['mkdHeaderTransparencyHeight'] = 0;
        }

        return $perPageVars;
    }
}