<?php
namespace MikadoCore\PostTypes\Portfolio\Shortcodes;

use MikadoCore\Lib;

/**
 * Class PortfolioList
 * @package MikadoCore\PostTypes\Portfolio\Shortcodes
 */
class PortfolioList implements Lib\ShortcodeInterface
{
    /**
     * @var string
     */
    private $base;

    public function __construct()
    {
        $this->base = 'mkd_portfolio_list';

        add_action('vc_before_init', array($this, 'vcMap'));
    }

    /**
     * Returns base for shortcode
     * @return string
     */
    public function getBase()
    {
        return $this->base;
    }

    /**
     * Maps shortcode to Visual Composer
     *
     * @see vc_map
     */
    public function vcMap()
    {
        if (function_exists('vc_map')) {

            $icons_array = array();
            if (mkd_core_theme_installed()) {
                $icons_array = \BuroMikadoIconCollections::get_instance()->getVCParamsArray();
            }

            vc_map(array(
                    'name'                      => esc_html__('Portfolio List', 'mikado-core'),
                    'base'                      => $this->getBase(),
                    'category'                  => esc_html__('by MIKADO', 'mikado-core'),
                    'icon'                      => 'icon-wpb-portfolio extended-custom-icon',
                    'allowed_container_element' => 'vc_row',
                    'params'                    => array(
                        array(
                            'type'        => 'dropdown',
                            'heading'     => esc_html__('Portfolio List Template', 'mikado-core'),
                            'param_name'  => 'type',
                            'value'       => array(
                                esc_html__('Standard', 'mikado-core')             => 'standard',
                                esc_html__('Gallery', 'mikado-core')              => 'gallery',
                                esc_html__('Masonry', 'mikado-core')              => 'masonry',
                                esc_html__('Pinterest', 'mikado-core')            => 'pinterest'
                            ),
                            'admin_label' => true
                        ),
                        array(
                            'type'       => 'dropdown',
                            'heading'    => esc_html__('Space Between Items', 'mikado-core'),
                            'param_name' => 'space_between_items',
                            'value'      => array(
                                ''                                   => '',
                                esc_html__('No Space', 'mikado-core') => 'no',
                                esc_html__('Small', 'mikado-core')    => 'small',
                                esc_html__('Medium', 'mikado-core')   => 'medium',
                                esc_html__('Large', 'mikado-core')    => 'large'
                            )
                        ),
                        array(
                            'type'       => 'dropdown',
                            'heading'    => esc_html__('Title Tag', 'mikado-core'),
                            'param_name' => 'title_tag',
                            'value'      => array(
                                ''   => '',
                                'h2' => 'h2',
                                'h3' => 'h3',
                                'h4' => 'h4',
                                'h5' => 'h5',
                                'h6' => 'h6',
                            )
                        ),
                        array(
                            'type'        => 'dropdown',
                            'heading'     => esc_html__('Image Proportions', 'mikado-core'),
                            'param_name'  => 'image_size',
                            'value'       => array(
                                esc_html__('Original', 'mikado-core')  => 'full',
                                esc_html__('Square', 'mikado-core')    => 'square',
                                esc_html__('Landscape', 'mikado-core') => 'landscape',
                                esc_html__('Portrait', 'mikado-core')  => 'portrait'
                            ),
                            'save_always' => true,
                            'dependency'  => array('element' => 'type', 'value' => array('standard', 'gallery', 'gallery-with-space'))
                        ),
                        array(
                            'type'        => 'dropdown',
                            'heading'     => esc_html__('Hover Type', 'mikado-core'),
                            'param_name'  => 'hover_type',
                            'value'       => array(
                                esc_html__('Zoom Lightbox', 'mikado-core')   => 'zoom-lightbox',
                                esc_html__('Zoom Out Simple Dark', 'mikado-core') => 'zoom-out-simple-dark',
                                esc_html__('Zoom Out Simple Light', 'mikado-core') => 'zoom-out-simple-light',
                                esc_html__('Content Always Visible', 'mikado-core')   => 'content-always-visible',
                                esc_html__('Follow', 'mikado-core')   => 'follow',
                            ),
                            'save_always' => true,
                            'dependency'  => array('element' => 'type', 'value' => array('gallery', 'gallery-with-space', 'masonry', 'masonry-with-space', 'pinterest', 'pinterest-with-space'))
                        ),
                        array(
                            'type'        => 'dropdown',
                            'heading'     => esc_html__('Appear Effect', 'mikado-core'),
                            'param_name'  => 'appear_effect',
                            'value'       => array(
                                esc_html__('None', 'mikado-core')  => '',
                                esc_html__('Column by Column', 'mikado-core')    => 'column-by-column',
                            ),
                            'dependency'  => array('element' => 'type', 'value' => array('pinterest'))
                        ),
                        array(
                            'type'        => 'dropdown',
                            'heading'     => esc_html__('Show More', 'mikado-core'),
                            'param_name'  => 'show_more',
                            'value'       => array(
                                esc_html__('Load More Button', 'mikado-core') => 'load_more',
                                esc_html__('Infinite Scroll', 'mikado-core')  => 'infinite_scroll',
                                esc_html__('None', 'mikado-core')             => 'none'
                            ),
                            'save_always' => true,
                            'description' => esc_html__('Default Value is Load More', 'mikado-core')
                        ),
                        array(
                            'type'        => 'dropdown',
                            'heading'     => esc_html__('Order By', 'mikado-core'),
                            'param_name'  => 'order_by',
                            'value'       => array(
                                esc_html__('Date', 'mikado-core')       => 'date',
                                esc_html__('Title', 'mikado-core')      => 'title',
                                esc_html__('Menu Order', 'mikado-core') => 'menu_order',
                            ),
                            'save_always' => true,
                            'group'       => esc_html__('Query and Layout Options', 'mikado-core')
                        ),
                        array(
                            'type'        => 'dropdown',
                            'heading'     => esc_html__('Order', 'mikado-core'),
                            'param_name'  => 'order',
                            'value'       => array(
                                esc_html__('ASC', 'mikado-core')  => 'ASC',
                                esc_html__('DESC', 'mikado-core') => 'DESC',
                            ),
                            'save_always' => true,
                            'group'       => esc_html__('Query and Layout Options', 'mikado-core')
                        ),
                        array(
                            'type'        => 'textfield',
                            'heading'     => esc_html__('One-Category Portfolio List', 'mikado-core'),
                            'param_name'  => 'category',
                            'description' => esc_html__('Enter one category slug (leave empty for showing all categories)', 'mikado-core'),
                            'group'       => esc_html__('Query and Layout Options', 'mikado-core')
                        ),
                        array(
                            'type'        => 'textfield',
                            'heading'     => esc_html__('Number of Portfolios Per Page', 'mikado-core'),
                            'param_name'  => 'number',
                            'value'       => '-1',
                            'description' => esc_html__('(enter -1 to show all)', 'mikado-core'),
                            'group'       => esc_html__('Query and Layout Options', 'mikado-core')
                        ),
                        array(
                            'type'        => 'dropdown',
                            'heading'     => esc_html__('Number of Columns', 'mikado-core'),
                            'param_name'  => 'columns',
                            'value'       => array(
                                ''                                => '',
                                esc_html__('One', 'mikado-core')   => '1',
                                esc_html__('Two', 'mikado-core')   => '2',
                                esc_html__('Three', 'mikado-core') => '3',
                                esc_html__('Four', 'mikado-core')  => '4',
                                esc_html__('Five', 'mikado-core')  => '5',
                                esc_html__('Six', 'mikado-core')   => '6'
                            ),
                            'description' => esc_html__('Default value is Three', 'mikado-core')
                        ),
                        array(
                            'type'        => 'textfield',
                            'heading'     => esc_html__('Show Only Projects with Listed IDs', 'mikado-core'),
                            'param_name'  => 'selected_projects',
                            'description' => esc_html__('Delimit ID numbers by comma (leave empty for all)', 'mikado-core'),
                            'group'       => esc_html__('Query and Layout Options', 'mikado-core')
                        ),
                        array(
                            'type'        => 'dropdown',
                            'heading'     => esc_html__('Enable Category Filter', 'mikado-core'),
                            'param_name'  => 'filter',
                            'value'       => array(
                                esc_html__('No', 'mikado-core')  => 'no',
                                esc_html__('Yes', 'mikado-core') => 'yes'
                            ),
                            'save_always' => true,
                            'description' => esc_html__('Default value is No', 'mikado-core')
                        ),
                        array(
                            'type'        => 'dropdown',
                            'heading'     => esc_html__('Filter Order By', 'mikado-core'),
                            'param_name'  => 'filter_order_by',
                            'value'       => array(
                                esc_html__('Name', 'mikado-core')  => 'name',
                                esc_html__('Count', 'mikado-core') => 'count',
                                esc_html__('Id', 'mikado-core')    => 'id',
                                esc_html__('Slug', 'mikado-core')  => 'slug'
                            ),
                            'save_always' => true,
                            'description' => esc_html__('Default value is Name', 'mikado-core'),
                            'dependency'  => array('element' => 'filter', 'value' => array('yes')),
                            'group'       => esc_html__('Query and Layout Options', 'mikado-core')
                        )
                    )
                )
            );
        }
    }

    /**
     * Renders shortcodes HTML
     *
     * @param $atts array of shortcode params
     * @param $content string shortcode content
     * @return string
     */
    public function render($atts, $content = null)
    {

        $args = array(
            'type'                => 'standard',
            'space_between_items' => 'medium',
            'columns'             => '3',
            'image_size'          => 'full',
            'hover_type'          => 'zoom-out-simple',
            'appear_effect'       => '',
            'order_by'            => 'date',
            'order'               => 'ASC',
            'number'              => '-1',
            'filter'              => 'no',
            'filter_order_by'     => 'name',
            'category'            => '',
            'selected_projects'   => '',
            'show_more'           => 'load_more',
            'title_tag'           => 'h3',
            'next_page'           => '',
            'portfolio_slider'    => '',
            'slider_behavior'     => '',
            'portfolios_shown'    => ''
        );

        $params = shortcode_atts($args, $atts);
        extract($params);

        $query_array = $this->getQueryArray($params);
        $query_results = new \WP_Query($query_array);
        $params['query_results'] = $query_results;
        $params['masonry_filter'] = '';

        $classes = $this->getPortfolioClasses($params);
        $data_atts = $this->getDataAtts($params);
        $data_atts .= 'data-max-num-pages = ' . $query_results->max_num_pages;

        $html = '';


        if ($filter == 'yes' && ($type == 'masonry' || $type == 'pinterest')) {
            $params['filter_categories'] = $this->getFilterCategories($params);
            $params['masonry_filter'] = 'mkd-masonry-filter';
            $html .= mkd_core_get_shortcode_module_template_part('portfolio', 'portfolio-filter', '', $params);
        }

        $html .= '<div class = "mkd-portfolio-list-holder-outer ' . $classes . '" ' . $data_atts . '>';

        if ($filter == 'yes' && ($type == 'standard' || $type == 'gallery')) {
            $params['filter_categories'] = $this->getFilterCategories($params);
            $html .= mkd_core_get_shortcode_module_template_part('portfolio', 'portfolio-filter', '', $params);
        }

        $html .= '<div class = "mkd-portfolio-list-holder clearfix" >';
        if ($type == 'masonry' || $type == 'pinterest') {
            $html .= '<div class="mkd-portfolio-list-masonry-grid-sizer"></div>';
            $html .= '<div class="mkd-portfolio-list-masonry-grid-gutter"></div>';
        }

        if ($query_results->have_posts()):
            while ($query_results->have_posts()) : $query_results->the_post();

                $params['current_id'] = get_the_ID();
                $params['thumb_size'] = $this->getImageSize($params);
                $params['category_html'] = $this->getItemCategoriesHtml($params);
                $params['categories'] = $this->getItemCategories($params);
                $params['article_masonry_size'] = $this->getMasonrySize($params);
                $params['item_link'] = $this->getItemLink($params);
                $params['hover_params']['category_html'] = $params['category_html'];
                $params['hover_params']['title_tag'] = $params['title_tag'];
                $params['hover_params']['item_link'] = $params['item_link'];

                $html .= mkd_core_get_shortcode_module_template_part('portfolio', $type, '', $params);

            endwhile;
        else:

            $html .= '<p>' . _e('Sorry, no posts matched your criteria.') . '</p>';

        endif;
        if ($portfolio_slider !== 'yes') {
            for ($i = 0; $i < (int)$columns; $i++) {
                $html .= "<div class='mkd-portfolio-gap'></div>\n";
            }
        }

        $html .= '</div>'; //close mkd-portfolio-list-holder
        if ($show_more !== 'none') {
            $html .= mkd_core_get_shortcode_module_template_part('portfolio', 'load-more-template', '', $params);
        }
        wp_reset_postdata();
        $html .= '</div>'; // close mkd-portfolio-list-holder-outer
        return $html;
    }

    /**
     * Generates portfolio list query attribute array
     *
     * @param $params
     *
     * @return array
     */
    public function getQueryArray($params)
    {

        $query_array = array();

        $query_array = array(
            'post_type'      => 'portfolio-item',
            'orderby'        => $params['order_by'],
            'order'          => $params['order'],
            'posts_per_page' => $params['number']
        );

        if (!empty($params['category'])) {
            $query_array['portfolio-category'] = $params['category'];
        }

        $project_ids = null;
        if (!empty($params['selected_projects'])) {
            $project_ids = explode(',', $params['selected_projects']);
            $query_array['post__in'] = $project_ids;
        }

        $paged = '';
        if (empty($params['next_page'])) {
            if (get_query_var('paged')) {
                $paged = get_query_var('paged');
            } elseif (get_query_var('page')) {
                $paged = get_query_var('page');
            }
        }

        if (!empty($params['next_page'])) {
            $query_array['paged'] = $params['next_page'];

        } else {
            $query_array['paged'] = 1;
        }

        return $query_array;
    }

    /**
     * Generates portfolio classes
     *
     * @param $params
     *
     * @return string
     */
    public function getPortfolioClasses($params)
    {
        $classes = array();
        $type = $params['type'];

        $hover_type = $params['hover_type'];
        $columns = $params['columns'];

        switch ($type):
            case 'standard':
                $classes[] = 'mkd-ptf-standard';
                break;
            case 'gallery':
                $classes[] = 'mkd-ptf-gallery';
                break;
            case 'masonry':
                $classes[] = 'mkd-ptf-masonry';
                break;
            case 'pinterest':
                $classes[] = 'mkd-ptf-pinterest';
                break;
        endswitch;


        //Space between items
        $classes[] = 'mkd-ptf-' . $params['space_between_items'] . '-space';

        if ($hover_type !== '' && $type !== 'standard') {
            $classes[] = 'mkd-ptf-hover-' . $hover_type;
        }

        if (!empty($params['appear_effect'])) {
            $classes[] = 'mkdf-ptf-appear-'. $params['appear_effect'];
        }

        if (empty($params['portfolio_slider'])) { // portfolio slider mustn't have this classes

            switch ($columns):
                case '1':
                    $classes[] = 'mkd-ptf-one-columns';
                    break;
                case '2':
                    $classes[] = 'mkd-ptf-two-columns';
                    break;
                case '3':
                    $classes[] = 'mkd-ptf-three-columns';
                    break;
                case '4':
                    $classes[] = 'mkd-ptf-four-columns';
                    break;
                case '5':
                    $classes[] = 'mkd-ptf-five-columns';
                    break;
                case '6':
                    $classes[] = 'mkd-ptf-six-columns';
                    break;
            endswitch;


            if ($params['show_more'] == 'load_more') {
                $classes[] = 'mkd-ptf-load-more mkd-ptf-show-more';
            } else if ($params['show_more'] == 'infinite_scroll') {
                $classes[] = 'mkd-ptf-infinite-scroll mkd-ptf-show-more';
            }

        }

        if ($params['filter'] == 'yes') {
            $classes[] = 'mkd-ptf-has-filter';
            if ($params['type'] == 'masonry' || $params['type'] == 'pinterest') {
                if ($params['filter'] == 'yes') {
                    $classes[] = 'mkd-ptf-masonry-filter';
                }
            }
        }

        if (!empty($params['portfolio_slider']) && $params['portfolio_slider'] == 'yes') {
            $classes[] = 'mkd-portfolio-slider-holder';

            if (!empty($params['slider_behavior'])) {
                $classes[] = 'mkd-ptf-slider-'. $params['slider_behavior'];
            }
        }

        return implode(' ', $classes);

    }

    /**
     * Generates portfolio image size
     *
     * @param $params
     *
     * @return string
     */
    public function getImageSize($params)
    {

        $thumb_size = 'full';
        $type = $params['type'];

        if ($type == 'standard' || $type == 'gallery') {
            if (!empty($params['image_size'])) {
                $image_size = $params['image_size'];

                switch ($image_size) {
                    case 'landscape':
                        $thumb_size = 'buro_mikado_landscape';
                        break;
                    case 'portrait':
                        $thumb_size = 'buro_mikado_portrait';
                        break;
                    case 'square':
                        $thumb_size = 'buro_mikado_square';
                        break;
                    case 'full':
                        $thumb_size = 'full';
                        break;
                }
            }
        } elseif ($type == 'masonry') {

            $id = $params['current_id'];
            $masonry_size = get_post_meta($id, 'portfolio_masonry_dimenisions', true);

            switch ($masonry_size):
                default :
                    $thumb_size = 'buro_mikado_square';
                    break;
                case 'large_width' :
                    $thumb_size = 'buro_mikado_large_width';
                    break;
                case 'large_height' :
                    $thumb_size = 'buro_mikado_large_height';
                    break;
                case 'large_width_height' :
                    $thumb_size = 'buro_mikado_large_width_height';
                    break;
            endswitch;
        }


        return $thumb_size;
    }

    /**
     * Generates portfolio item categories ids.This function is used for filtering
     *
     * @param $params
     *
     * @return array
     */
    public function getItemCategories($params)
    {
        $id = $params['current_id'];
        $category_return_array = array();

        $categories = wp_get_post_terms($id, 'portfolio-category');

        foreach ($categories as $cat) {
            $category_return_array[] = 'portfolio_category_' . $cat->term_id;
        }
        return implode(' ', $category_return_array);
    }

    /**
     * Generates portfolio item categories html based on id
     *
     * @param $params
     *
     * @return html
     */
    public function getItemCategoriesHtml($params)
    {
        $id = $params['current_id'];

        $categories = wp_get_post_terms($id, 'portfolio-category');
        $category_html = '<div class="mkd-ptf-category-holder">';
        $k = 1;

        foreach ($categories as $cat) {
            $category_html .= '<div class="mkd-ptf-category-inner"><h6 class="mkd-ptf-category">' . $cat->name . '</h6></div>';
            if (count($categories) != $k) {
                $category_html .= ' , ';
            }
            $k++;
        }
        $category_html .= '</div>';
        return $category_html;
    }

    /**
     * Generates masonry size class for each article( based on id)
     *
     * @param $params
     *
     * @return string
     */
    public function getMasonrySize($params)
    {
        $masonry_size_class = '';

        if ($params['type'] == 'masonry') {

            $id = $params['current_id'];
            $masonry_size = get_post_meta($id, 'portfolio_masonry_dimenisions', true);
            switch ($masonry_size):
                default :
                    $masonry_size_class = 'mkd-default-masonry-item';
                    break;
                case 'large_width' :
                    $masonry_size_class = 'mkd-large-width-masonry-item';
                    break;
                case 'large_height' :
                    $masonry_size_class = 'mkd-large-height-masonry-item';
                    break;
                case 'large_width_height' :
                    $masonry_size_class = 'mkd-large-width-height-masonry-item';
                    break;
            endswitch;
        }

        return $masonry_size_class;
    }

    /**
     * Generates filter categories array
     *
     * @param $params
     *
     * * @return array
     */
    public function getFilterCategories($params)
    {

        $cat_id = 0;
        $top_category = '';

        if (!empty($params['category'])) {

            $top_category = get_term_by('slug', $params['category'], 'portfolio-category');
            if (isset($top_category->term_id)) {
                $cat_id = $top_category->term_id;
            }

        }

        $args = array(
            'taxonomy' => 'portfolio-category',
            'child_of' => $cat_id,
            'order_by' => $params['filter_order_by']
        );

        $filter_categories = get_terms($args);

        return $filter_categories;

    }

    /**
     * Generates datta attributes array
     *
     * @param $params
     *
     * @return array
     */
    public function getDataAtts($params)
    {

        $data_attr = array();
        $data_return_string = '';

        if (get_query_var('paged')) {
            $paged = get_query_var('paged');
        } elseif (get_query_var('page')) {
            $paged = get_query_var('page');
        } else {
            $paged = 1;
        }

        if (!empty($paged)) {
            $data_attr['data-next-page'] = $paged + 1;
        }
        if (!empty($params['type'])) {
            $data_attr['data-type'] = $params['type'];
        }
        if (!empty($params['columns'])) {
            $data_attr['data-columns'] = $params['columns'];
        }
        if (!empty($params['order_by'])) {
            $data_attr['data-order-by'] = $params['order_by'];
        }
        if (!empty($params['order'])) {
            $data_attr['data-order'] = $params['order'];
        }
        if (!empty($params['number'])) {
            $data_attr['data-number'] = $params['number'];
        }
        if (!empty($params['image_size'])) {
            $data_attr['data-image-size'] = $params['image_size'];
        }
        if (!empty($params['hover_type'])) {
            $data_attr['data-hover-type'] = $params['hover_type'];
        }
        if (!empty($params['filter'])) {
            $data_attr['data-filter'] = $params['filter'];
        }
        if (!empty($params['filter_order_by'])) {
            $data_attr['data-filter-order-by'] = $params['filter_order_by'];
        }
        if (!empty($params['category'])) {
            $data_attr['data-category'] = $params['category'];
        }
        if (!empty($params['selected_projects'])) {
            $data_attr['data-selected-projects'] = $params['selected_projects'];
        }
        if (!empty($params['show_more'])) {
            $data_attr['data-show-more'] = $params['show_more'];
        }
        if (!empty($params['title_tag'])) {
            $data_attr['data-title-tag'] = $params['title_tag'];
        }
        if (!empty($params['portfolio_slider']) && $params['portfolio_slider'] == 'yes') {
            $data_attr['data-items'] = $params['portfolios_shown'];
        }

        foreach ($data_attr as $key => $value) {
            if ($key !== '') {
                $data_return_string .= $key . '= "' . esc_attr($value) . '" ';
            }
        }
        return $data_return_string;
    }

    public function getItemLink($params)
    {

        $id = $params['current_id'];
        $portfolio_link = get_permalink($id);
        if (get_post_meta($id, 'portfolio_external_link', true) !== '') {
            $portfolio_link = get_post_meta($id, 'portfolio_external_link', true);
        }

        return $portfolio_link;

    }
}